/*
 * Decompiled with CFR 0.152.
 */
package mage.game.match;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import mage.cards.decks.Deck;
import mage.game.Game;
import mage.game.GameException;
import mage.game.GameInfo;
import mage.game.events.Listener;
import mage.game.events.TableEvent;
import mage.game.events.TableEventSource;
import mage.game.match.Match;
import mage.game.match.MatchOptions;
import mage.game.match.MatchPlayer;
import mage.game.result.ResultProtos;
import mage.players.Player;
import mage.util.DateFormat;
import mage.util.RandomUtil;
import mage.util.ThreadUtils;
import org.apache.log4j.Logger;

public abstract class MatchImpl
implements Match {
    private static final Logger logger = Logger.getLogger(MatchImpl.class);
    protected UUID id = UUID.randomUUID();
    protected List<MatchPlayer> players = new ArrayList<MatchPlayer>();
    protected List<Game> games = new ArrayList<Game>();
    protected List<GameInfo> gamesInfo = new ArrayList<GameInfo>();
    protected UUID tableId;
    protected MatchOptions options;
    protected TableEventSource tableEventSource = new TableEventSource();
    protected Date startTime;
    protected Date endTime;
    protected int draws;
    protected int startedGames;
    protected boolean replayAvailable;

    public MatchImpl(MatchOptions options) {
        this.options = options;
        this.startTime = new Date();
        this.replayAvailable = false;
        this.draws = 0;
    }

    @Override
    public List<MatchPlayer> getPlayers() {
        return this.players;
    }

    @Override
    public MatchPlayer getPlayer(UUID playerId) {
        for (MatchPlayer player : this.players) {
            if (!player.getPlayer().getId().equals(playerId)) continue;
            return player;
        }
        return null;
    }

    @Override
    public void addPlayer(Player player, Deck deck) {
        MatchPlayer matchPlayer = new MatchPlayer(player, deck, this);
        player.setMatchPlayer(matchPlayer);
        this.players.add(matchPlayer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean quitMatch(UUID playerId) {
        MatchPlayer mPlayer = this.getPlayer(playerId);
        if (mPlayer != null) {
            if (!this.hasStarted()) {
                return this.players.remove(mPlayer);
            }
            mPlayer.setQuit(true);
            MatchImpl matchImpl = this;
            synchronized (matchImpl) {
                this.notifyAll();
            }
            this.checkIfMatchEnds();
            return true;
        }
        return false;
    }

    @Override
    public void startMatch() {
        this.startTime = new Date();
    }

    @Override
    public UUID getId() {
        return this.id;
    }

    @Override
    public String getName() {
        return this.options.getName();
    }

    @Override
    public MatchOptions getOptions() {
        return this.options;
    }

    @Override
    public boolean hasEnded() {
        if (this.getGame() == null && this.isDoneSideboarding()) {
            this.checkIfMatchEnds();
        }
        if (this.getGame() != null && this.getGame().hasEnded()) {
            for (MatchPlayer matchPlayer : this.players) {
                if (!matchPlayer.getPlayer().hasQuit() || matchPlayer.hasQuit()) continue;
                logger.warn((Object)("MatchPlayer was not set to quit matchId " + this.getId() + " - " + matchPlayer.getName()));
                matchPlayer.setQuit(true);
            }
            this.checkIfMatchEnds();
        }
        return this.endTime != null;
    }

    @Override
    public boolean hasStarted() {
        return this.startedGames > 0;
    }

    @Override
    public boolean checkIfMatchEnds() {
        int activePlayers = 0;
        MatchPlayer matchWinner = null;
        for (MatchPlayer matchPlayer : this.players) {
            if (!matchPlayer.hasQuit()) {
                ++activePlayers;
                matchWinner = matchPlayer;
            }
            if (matchPlayer.getWins() < this.options.getWinsNeeded()) continue;
            matchPlayer.setMatchWinner(true);
            this.endTime = new Date();
            return true;
        }
        if (activePlayers < 2) {
            if (matchWinner != null) {
                matchWinner.setMatchWinner(true);
            }
            this.endTime = new Date();
            return true;
        }
        return false;
    }

    @Override
    public Game getGame() {
        if (this.games.isEmpty()) {
            return null;
        }
        return this.games.get(this.games.size() - 1);
    }

    @Override
    public List<Game> getGames() {
        return this.games;
    }

    @Override
    public void addGame() {
        ++this.startedGames;
    }

    @Override
    public int getNumGames() {
        return this.startedGames;
    }

    @Override
    public int getWinsNeeded() {
        return this.options.getWinsNeeded();
    }

    @Override
    public int getFreeMulligans() {
        return this.options.getFreeMulligans();
    }

    protected void initGame(Game game) throws GameException {
        game.setTableId(this.tableId);
        this.addGame();
        this.shufflePlayers();
        for (MatchPlayer matchPlayer : this.players) {
            if (!matchPlayer.hasQuit() && matchPlayer.getDeck() != null) {
                matchPlayer.getPlayer().init(game);
                game.loadCards(matchPlayer.getDeck().getCards(), matchPlayer.getPlayer().getId());
                game.loadCards(matchPlayer.getDeck().getSideboard(), matchPlayer.getPlayer().getId());
                game.addPlayer(matchPlayer.getPlayer(), matchPlayer.getDeck());
                matchPlayer.getPlayer().setBufferTimeLeft(this.options.getMatchBufferTime().getBufferSecs());
                if (this.games.isEmpty()) {
                    matchPlayer.getPlayer().setPriorityTimeLeft(this.options.getMatchTimeLimit().getPrioritySecs());
                    continue;
                }
                if (matchPlayer.getPriorityTimeLeft() <= 0) continue;
                matchPlayer.getPlayer().setPriorityTimeLeft(matchPlayer.getPriorityTimeLeft());
                continue;
            }
            if (matchPlayer.getDeck() != null) continue;
            logger.error((Object)("Match: " + this.getId() + " " + matchPlayer.getName() + " has no deck."));
        }
        game.setPriorityTime(this.options.getMatchTimeLimit().getPrioritySecs());
        game.setBufferTime(this.options.getMatchBufferTime().getBufferSecs());
    }

    protected void shufflePlayers() {
        Collections.shuffle(this.players, RandomUtil.getRandom());
    }

    @Override
    public void endGame() {
        Game game = this.getGame();
        for (MatchPlayer matchPlayer : this.players) {
            Player player = game.getPlayer(matchPlayer.getPlayer().getId());
            if (player == null) continue;
            if (game.getPriorityTime() > 0) {
                matchPlayer.setPriorityTimeLeft(player.getPriorityTimeLeft());
            }
            if (player.hasQuit()) {
                matchPlayer.setQuit(true);
            }
            if (!player.hasWon()) continue;
            matchPlayer.addWin();
        }
        if (game.isADraw()) {
            this.addDraw();
        }
        this.checkIfMatchEnds();
        game.fireGameEndInfo();
        this.gamesInfo.add(this.createGameInfo(game));
    }

    @Override
    public GameInfo createGameInfo(Game game) {
        String result;
        String state;
        StringBuilder playersInfo = new StringBuilder();
        int counter = 0;
        for (MatchPlayer matchPlayer : this.getPlayers()) {
            if (counter > 0) {
                playersInfo.append(" - ");
            }
            playersInfo.append(matchPlayer.getName());
            ++counter;
        }
        String duelingTime = "";
        if (game.hasEnded()) {
            if (game.getEndTime() != null) {
                duelingTime = " (" + DateFormat.getDuration((game.getEndTime().getTime() - game.getStartTime().getTime()) / 1000L) + ')';
            }
            state = "Finished" + duelingTime;
            result = game.getWinner();
        } else {
            if (game.getStartTime() != null) {
                duelingTime = " (" + DateFormat.getDuration((new Date().getTime() - game.getStartTime().getTime()) / 1000L) + ')';
            }
            state = "Dueling" + duelingTime;
            result = "";
        }
        return new GameInfo(0, this.getId(), game.getId(), state, result, playersInfo.toString(), this.tableId);
    }

    @Override
    public List<GameInfo> getGamesInfo() {
        return this.gamesInfo;
    }

    @Override
    public void setTableId(UUID tableId) {
        this.tableId = tableId;
        this.games.forEach(game -> game.setTableId(tableId));
    }

    @Override
    public void setTournamentRound(int round) {
        for (GameInfo gameInfo : this.gamesInfo) {
            gameInfo.setRoundNum(round);
        }
    }

    @Override
    public UUID getChooser() {
        UUID loserId = null;
        Game game = this.getGame();
        for (MatchPlayer player : this.players) {
            Player p = game.getPlayer(player.getPlayer().getId());
            if (p == null || !p.hasLost() || p.hasQuit()) continue;
            loserId = p.getId();
        }
        return loserId;
    }

    @Override
    public void addTableEventListener(Listener<TableEvent> listener) {
        this.tableEventSource.addListener(listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void sideboard() {
        ThreadUtils.ensureRunInGameThread();
        for (MatchPlayer player : this.players) {
            if (player.hasQuit()) continue;
            if (player.getDeck() != null) {
                player.setSideboarding();
                player.getPlayer().sideboard(this, player.getDeck());
                continue;
            }
            logger.error((Object)("Player " + player.getName() + " has no deck: " + player.getPlayer().getId()));
        }
        MatchImpl matchImpl = this;
        synchronized (matchImpl) {
            while (!this.isDoneSideboarding()) {
                try {
                    this.wait();
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }

    @Override
    public boolean isDoneSideboarding() {
        for (MatchPlayer player : this.players) {
            if (player.hasQuit() || player.isDoneSideboarding()) continue;
            return false;
        }
        return true;
    }

    public boolean areAllDoneSideboarding() {
        int count = 0;
        for (MatchPlayer player : this.players) {
            if (!player.hasQuit() && player.isDoneSideboarding()) {
                return true;
            }
            if (!player.hasQuit()) continue;
            ++count;
        }
        return count < this.players.size();
    }

    @Override
    public void fireSideboardEvent(UUID playerId, Deck deck) {
        MatchPlayer player = this.getPlayer(playerId);
        if (player != null) {
            this.tableEventSource.fireTableEvent(TableEvent.EventType.SIDEBOARD, playerId, deck, 180);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void submitDeck(UUID playerId, Deck deck) {
        MatchPlayer player = this.getPlayer(playerId);
        if (player != null) {
            deck.setName(player.getDeck().getName());
            player.submitDeck(deck);
        }
        MatchImpl matchImpl = this;
        synchronized (matchImpl) {
            this.notifyAll();
        }
    }

    @Override
    public void updateDeck(UUID playerId, Deck deck, boolean ignoreMainBasicLands) {
        MatchPlayer player = this.getPlayer(playerId);
        if (player == null) {
            return;
        }
        player.updateDeck(deck, ignoreMainBasicLands);
    }

    protected String createGameStartMessage() {
        StringBuilder sb = new StringBuilder();
        sb.append("<br/><b>Match score:</b><br/>");
        for (MatchPlayer mp : this.getPlayers()) {
            sb.append("   ").append(mp.getPlayer().getLogName());
            sb.append(" - ").append(mp.getWins()).append(mp.getWins() == 1 ? " win" : " wins");
            if (mp.hasQuit()) {
                sb.append(" QUITTED");
            }
            sb.append("<br/>");
        }
        if (this.getDraws() > 0) {
            sb.append("   Draws: ").append(this.getDraws()).append("<br/>");
        }
        if (this.options.getRange() != null) {
            sb.append("   Range: ").append(this.options.getRange().toString()).append("<br/>");
        }
        sb.append("   Mulligan type: ").append(this.options.getMulliganType().toString()).append("<br/>");
        sb.append("   Free mulligans: ").append(this.options.getFreeMulligans()).append("<br/>");
        if (this.options.isPlaneChase()) {
            sb.append("   Planechase mode activated").append("<br/>");
        }
        sb.append("<br/>");
        sb.append("Match is ").append(this.getOptions().isRated() ? "" : "not ").append("rated<br/>");
        sb.append("You have to win ").append(this.getWinsNeeded()).append(this.getWinsNeeded() == 1 ? " game" : " games").append(" to win the complete match<br/>");
        sb.append("<br/>Game has started<br/><br/>");
        return sb.toString();
    }

    @Override
    public Date getStartTime() {
        if (this.startTime != null) {
            return new Date(this.startTime.getTime());
        }
        return null;
    }

    @Override
    public Date getEndTime() {
        if (this.endTime != null) {
            return new Date(this.endTime.getTime());
        }
        return null;
    }

    @Override
    public boolean isReplayAvailable() {
        return this.replayAvailable;
    }

    @Override
    public void setReplayAvailable(boolean replayAvailable) {
        this.replayAvailable = replayAvailable;
    }

    @Override
    public void cleanUpOnMatchEnd(boolean isSaveGameActivated, boolean isTournament) {
        for (MatchPlayer matchPlayer : this.players) {
            matchPlayer.cleanUpOnMatchEnd();
        }
        if (!isSaveGameActivated && !isTournament || this.getGame().isSimulation()) {
            this.getGames().clear();
        }
    }

    @Override
    public void addDraw() {
        ++this.draws;
    }

    @Override
    public int getDraws() {
        return this.draws;
    }

    @Override
    public void cleanUp() {
        for (MatchPlayer matchPlayer : this.players) {
            matchPlayer.cleanUpOnMatchEnd();
        }
        this.getGames().clear();
    }

    @Override
    public ResultProtos.MatchProto toProto() {
        ResultProtos.MatchProto.Builder builder = ResultProtos.MatchProto.newBuilder().setName(this.getName()).setGameType(this.getOptions().getGameType()).setDeckType(this.getOptions().getDeckType()).setGames(this.getNumGames()).setDraws(this.getDraws()).setMatchOptions(this.getOptions().toProto()).setEndTimeMs((this.getEndTime() != null ? this.getEndTime() : new Date()).getTime());
        for (MatchPlayer matchPlayer : this.getPlayers()) {
            ResultProtos.MatchQuitStatus status = !matchPlayer.hasQuit() ? ResultProtos.MatchQuitStatus.NO_MATCH_QUIT : (matchPlayer.getPlayer().hasTimerTimeout() ? ResultProtos.MatchQuitStatus.TIMER_TIMEOUT : (matchPlayer.getPlayer().hasIdleTimeout() ? ResultProtos.MatchQuitStatus.IDLE_TIMEOUT : ResultProtos.MatchQuitStatus.QUIT));
            builder.addPlayersBuilder().setName(matchPlayer.getName()).setHuman(matchPlayer.getPlayer().isHuman()).setQuit(status).setWins(matchPlayer.getWins());
        }
        return builder.build();
    }
}

