/*
 * Decompiled with CFR 0.152.
 */
package mage.server;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import mage.MageException;
import mage.cards.decks.Deck;
import mage.cards.decks.DeckCardLists;
import mage.cards.decks.DeckValidatorError;
import mage.cards.decks.DeckValidatorFactory;
import mage.constants.RangeOfInfluence;
import mage.constants.TableState;
import mage.game.Game;
import mage.game.GameException;
import mage.game.GameOptions;
import mage.game.Seat;
import mage.game.Table;
import mage.game.draft.Draft;
import mage.game.draft.DraftPlayer;
import mage.game.match.Match;
import mage.game.match.MatchOptions;
import mage.game.match.MatchPlayer;
import mage.game.tournament.Tournament;
import mage.game.tournament.TournamentOptions;
import mage.game.tournament.TournamentPlayer;
import mage.players.Player;
import mage.players.PlayerType;
import mage.server.Main;
import mage.server.User;
import mage.server.game.GameFactory;
import mage.server.game.PlayerFactory;
import mage.server.managers.ManagerFactory;
import mage.server.record.TableRecorderImpl;
import mage.server.tournament.TournamentFactory;
import mage.server.util.ServerMessagesUtil;
import mage.view.ChatMessage;
import org.apache.log4j.Logger;

public class TableController {
    private static final Logger logger = Logger.getLogger(TableController.class);
    private final ManagerFactory managerFactory;
    private final UUID userId;
    private final UUID chatId;
    private final String controllerName;
    private final Table table;
    private final ConcurrentHashMap<UUID, UUID> userPlayerMap = new ConcurrentHashMap();
    private Match match;
    private MatchOptions options;
    private Tournament tournament;
    private ScheduledFuture<?> futureTimeout;
    protected final ScheduledExecutorService timeoutExecutor;

    public TableController(ManagerFactory managerFactory, UUID roomId, UUID userId, MatchOptions options) {
        this.managerFactory = managerFactory;
        this.timeoutExecutor = managerFactory.threadExecutor().getTimeoutExecutor();
        this.userId = userId;
        this.options = options;
        this.match = GameFactory.instance.createMatch(options.getGameType(), options);
        if (userId != null) {
            Optional<User> user = managerFactory.userManager().getUser(userId);
            this.controllerName = user.map(User::getName).orElse("undefined");
        } else {
            this.controllerName = "System";
        }
        this.table = new Table(roomId, options.getGameType(), options.getName(), this.controllerName, DeckValidatorFactory.instance.createDeckValidator(options.getDeckType()), options.getPlayerTypes(), (Table.TableRecorder)new TableRecorderImpl(managerFactory.userManager()), this.match, options.getBannedUsers(), options.isPlaneChase());
        this.chatId = managerFactory.chatManager().createTableChatSession(this.table);
        this.init();
    }

    public TableController(ManagerFactory managerFactory, UUID roomId, UUID userId, TournamentOptions options) {
        this.managerFactory = managerFactory;
        this.timeoutExecutor = managerFactory.threadExecutor().getTimeoutExecutor();
        this.userId = userId;
        this.tournament = TournamentFactory.instance.createTournament(options.getTournamentType(), options);
        if (userId != null) {
            Optional<User> user = managerFactory.userManager().getUser(userId);
            if (!user.isPresent()) {
                logger.fatal((Object)("User for userId " + userId + " could not be retrieved from UserManagerImpl"));
                this.controllerName = "[unknown]";
            } else {
                this.controllerName = user.get().getName();
            }
        } else {
            this.controllerName = "System";
        }
        this.table = new Table(roomId, options.getTournamentType(), options.getName(), this.controllerName, DeckValidatorFactory.instance.createDeckValidator(options.getMatchOptions().getDeckType()), options.getPlayerTypes(), (Table.TableRecorder)new TableRecorderImpl(managerFactory.userManager()), this.tournament, options.getMatchOptions().getBannedUsers(), options.isPlaneChase());
        this.chatId = managerFactory.chatManager().createTableChatSession(this.table);
    }

    private void init() {
        this.match.addTableEventListener(event -> {
            try {
                switch (event.getEventType()) {
                    case SIDEBOARD: {
                        this.sideboard(event.getPlayerId(), event.getDeck());
                    }
                }
            }
            catch (MageException ex) {
                logger.fatal((Object)"Table event listener error", (Throwable)ex);
            }
        });
    }

    public synchronized boolean joinTournament(UUID userId, String name, PlayerType playerType, int skill, DeckCardLists deckList, String password) throws GameException {
        int quitRatio;
        if (this.table.getState() != TableState.WAITING) {
            return false;
        }
        Seat seat = this.table.getNextAvailableSeat(playerType);
        if (seat == null) {
            throw new GameException("No available seats.");
        }
        Optional<User> _user = this.managerFactory.userManager().getUser(userId);
        if (!_user.isPresent()) {
            logger.fatal((Object)("couldn't get user " + name + " for join tournament userId = " + userId));
            return false;
        }
        User user = _user.get();
        if (!this.table.getTournament().getOptions().getPassword().isEmpty() && playerType == PlayerType.HUMAN && !this.table.getTournament().getOptions().getPassword().equals(password)) {
            user.showUserMessage("Join Table", "Wrong password.");
            return false;
        }
        if (this.userPlayerMap.containsKey(userId) && playerType == PlayerType.HUMAN) {
            user.showUserMessage("Join Table", "You can join a table only one time.");
            return false;
        }
        Deck deck = null;
        if (!this.table.getTournament().getTournamentType().isLimited()) {
            if (deckList == null) {
                user.showUserMessage("Join Table", "No valid deck selected!");
                return false;
            }
            deck = Deck.load(deckList, false, false);
            if (!Main.isTestMode() && !this.table.getValidator().validate(deck)) {
                StringBuilder sb = new StringBuilder("You (").append(name).append(") have an invalid deck for the selected ").append(this.table.getValidator().getName()).append(" Format. \n\n");
                List<DeckValidatorError> errorsList = this.table.getValidator().getErrorsListSorted();
                errorsList.stream().forEach(error -> sb.append(error.getGroup()).append(": ").append(error.getMessage()).append("\n"));
                sb.append("\n\nSelect a deck that is appropriate for the selected format and try again!");
                user.showUserMessage("Join Table", sb.toString());
                if (this.isOwner(userId)) {
                    logger.debug((Object)("New table removed because owner submitted invalid deck tableId " + this.table.getId()));
                    this.managerFactory.tableManager().removeTable(this.table.getId());
                }
                return false;
            }
        }
        if ((quitRatio = this.table.getTournament().getOptions().getQuitRatio()) < user.getTourneyQuitRatio()) {
            String message = "Your quit ratio " + user.getTourneyQuitRatio() + "% is higher than the table requirement " + quitRatio + '%';
            user.showUserMessage("Join Table", message);
            return false;
        }
        int minimumRating = this.table.getTournament().getOptions().getMinimumRating();
        int userRating = this.table.getTournament().getOptions().getMatchOptions().isLimited() ? user.getUserData().getLimitedRating() : user.getUserData().getConstructedRating();
        if (userRating < minimumRating) {
            String message = "Your rating " + userRating + " is lower than the table requirement " + minimumRating;
            user.showUserMessage("Join Table", message);
            return false;
        }
        Optional<Player> playerOptional = this.createPlayer(name, seat.getPlayerType(), skill);
        if (playerOptional.isPresent()) {
            Player player = playerOptional.get();
            if (!player.canJoinTable(this.table)) {
                user.showUserMessage("Join Table", "A " + (Object)((Object)seat.getPlayerType()) + " player can't join this table.");
                return false;
            }
            this.tournament.addPlayer(player, seat.getPlayerType());
            TournamentPlayer tournamentPlayer = this.tournament.getPlayer(player.getId());
            if (deck != null && tournamentPlayer != null) {
                tournamentPlayer.submitDeck(deck);
            }
            this.table.joinTable(player, seat);
            logger.debug((Object)("Player " + player.getName() + " id: " + player.getId() + " joined tableId: " + this.table.getId()));
            if (seat.getPlayer().isHuman()) {
                seat.getPlayer().setUserData(user.getUserData());
                user.addTable(player.getId(), this.table);
                user.ccJoinedTable(this.table.getRoomId(), this.table.getId(), this.table.getParentTableId(), true);
                this.userPlayerMap.put(userId, player.getId());
            }
            return true;
        }
        throw new GameException("Playertype " + (Object)((Object)seat.getPlayerType()) + " could not be created.");
    }

    public boolean hasPlayer(UUID userId) {
        return this.userPlayerMap.containsKey(userId);
    }

    public synchronized boolean replaceDraftPlayer(Player oldPlayer, String name, PlayerType playerType, int skill) {
        Optional<Player> newPlayerOpt = this.createPlayer(name, playerType, skill);
        if (!newPlayerOpt.isPresent() || this.table.getState() != TableState.DRAFTING) {
            return false;
        }
        Player newPlayer = newPlayerOpt.get();
        TournamentPlayer oldTournamentPlayer = this.tournament.getPlayer(oldPlayer.getId());
        this.tournament.removePlayer(oldPlayer.getId());
        this.tournament.addPlayer(newPlayer, playerType);
        TournamentPlayer newTournamentPlayer = this.tournament.getPlayer(newPlayer.getId());
        newTournamentPlayer.setState(oldTournamentPlayer.getState());
        newTournamentPlayer.setReplacedTournamentPlayer(oldTournamentPlayer);
        this.managerFactory.draftManager().getController(this.table.getId()).ifPresent(controller -> controller.replacePlayer(oldPlayer, newPlayer));
        return true;
    }

    public synchronized boolean joinTable(UUID userId, String name, PlayerType playerType, int skill, DeckCardLists deckList, String password) throws MageException {
        Optional<Player> playerOpt;
        Optional<User> _user = this.managerFactory.userManager().getUser(userId);
        if (!_user.isPresent()) {
            logger.error((Object)("Join Table: can't find user to join " + name + " Id = " + userId));
            return false;
        }
        User user = _user.get();
        if (this.userPlayerMap.containsKey(userId) && playerType == PlayerType.HUMAN) {
            user.showUserMessage("Join Table", "You can join a table only one time.");
            return false;
        }
        if (this.table.getState() != TableState.WAITING) {
            user.showUserMessage("Join Table", "No available seats.");
            return false;
        }
        if (!this.table.getMatch().getOptions().getPassword().isEmpty() && playerType == PlayerType.HUMAN && !this.table.getMatch().getOptions().getPassword().equals(password)) {
            user.showUserMessage("Join Table", "Wrong password.");
            return false;
        }
        Seat seat = this.table.getNextAvailableSeat(playerType);
        if (seat == null) {
            user.showUserMessage("Join Table", "No available seats.");
            return false;
        }
        Deck deck = Deck.load(deckList, false, false);
        if (!Main.isTestMode() && !this.table.getValidator().validate(deck)) {
            StringBuilder sb = new StringBuilder("You (").append(name).append(") have an invalid deck for the selected ").append(this.table.getValidator().getName()).append(" Format. \n\n");
            List<DeckValidatorError> errorsList = this.table.getValidator().getErrorsListSorted();
            errorsList.stream().forEach(error -> sb.append(error.getGroup()).append(": ").append(error.getMessage()).append("\n"));
            sb.append("\n\nSelect a deck that is appropriate for the selected format and try again!");
            user.showUserMessage("Join Table", sb.toString());
            if (this.isOwner(userId)) {
                logger.debug((Object)("New table removed because owner submitted invalid deck tableId " + this.table.getId()));
                this.managerFactory.tableManager().removeTable(this.table.getId());
            }
            return false;
        }
        int quitRatio = this.table.getMatch().getOptions().getQuitRatio();
        if (quitRatio < user.getMatchQuitRatio()) {
            String message = "Your quit ratio " + user.getMatchQuitRatio() + "% is higher than the table requirement " + quitRatio + '%';
            user.showUserMessage("Join Table", message);
            return false;
        }
        int minimumRating = this.table.getMatch().getOptions().getMinimumRating();
        int userRating = this.table.getMatch().getOptions().isLimited() ? user.getUserData().getLimitedRating() : user.getUserData().getConstructedRating();
        if (userRating < minimumRating) {
            String message = "Your rating " + userRating + " is lower than the table requirement " + minimumRating;
            user.showUserMessage("Join Table", message);
            return false;
        }
        int edhPowerLevel = this.table.getMatch().getOptions().getEdhPowerLevel();
        if (edhPowerLevel > 0 && this.table.getValidator().getName().toLowerCase(Locale.ENGLISH).equals("commander")) {
            int deckEdhPowerLevel = this.table.getValidator().getEdhPowerLevel(deck, new ArrayList<String>(), new ArrayList<String>());
            if (deckEdhPowerLevel % 100 > edhPowerLevel) {
                String message = "Your deck appears to be too powerful for this table.\n\nReduce the number of extra turn cards, infect, counters, fogs, reconsider your commander. " + "\nThe table requirement has a maximum power level of " + edhPowerLevel + " whilst your deck has a calculated power level of " + deckEdhPowerLevel % 100;
                user.showUserMessage("Join Table", message);
                return false;
            }
            boolean restrictedColor = false;
            String badColor = "";
            int colorVal = edhPowerLevel % 10;
            if (colorVal == 6 && deckEdhPowerLevel >= 10000000) {
                restrictedColor = true;
                badColor = "white";
            }
            if (colorVal == 4 && deckEdhPowerLevel % 10000000 >= 1000000) {
                restrictedColor = true;
                badColor = "blue";
            }
            if (colorVal == 3 && deckEdhPowerLevel % 1000000 >= 100000) {
                restrictedColor = true;
                badColor = "black";
            }
            if (colorVal == 2 && deckEdhPowerLevel % 100000 >= 10000) {
                restrictedColor = true;
                badColor = "red";
            }
            if (colorVal == 1 && deckEdhPowerLevel % 10000 >= 1000) {
                restrictedColor = true;
                badColor = "green";
            }
            if (restrictedColor) {
                String message = "Your deck contains " + badColor + ".  The creator of the table has requested no " + badColor + " cards to be on the table!";
                user.showUserMessage("Join Table", message);
                return false;
            }
        }
        if (!(playerOpt = this.createPlayer(name, seat.getPlayerType(), skill)).isPresent()) {
            String message = "Could not create player " + name + " of type " + (Object)((Object)seat.getPlayerType());
            logger.warn((Object)("User: " + user.getName() + " => " + message));
            user.showUserMessage("Join Table", message);
            return false;
        }
        Player player = playerOpt.get();
        if (!player.canJoinTable(this.table)) {
            user.showUserMessage("Join Table", "A " + (Object)((Object)seat.getPlayerType()) + " player can't join this table.");
            return false;
        }
        this.match.addPlayer(player, deck);
        this.table.joinTable(player, seat);
        if (seat.getPlayer().isHuman()) {
            seat.getPlayer().setUserData(user.getUserData());
            if (!this.table.isTournamentSubTable()) {
                user.addTable(player.getId(), this.table);
            }
            user.ccJoinedTable(this.table.getRoomId(), this.table.getId(), this.table.getParentTableId(), false);
            this.userPlayerMap.put(userId, player.getId());
        }
        return true;
    }

    public void addPlayer(UUID userId, Player player, PlayerType playerType, Deck deck) throws GameException {
        if (this.table.getState() != TableState.WAITING) {
            return;
        }
        Seat seat = this.table.getNextAvailableSeat(playerType);
        if (seat == null) {
            throw new GameException("No available seats.");
        }
        this.match.addPlayer(player, deck);
        this.table.joinTable(player, seat);
        if (player.isHuman()) {
            this.userPlayerMap.put(userId, player.getId());
        }
    }

    public synchronized boolean submitDeck(UUID userId, DeckCardLists deckList) throws MageException {
        MatchPlayer mPlayer;
        MatchPlayer mPlayer2;
        TournamentPlayer player;
        TournamentPlayer tournamentPlayer;
        UUID playerId = this.userPlayerMap.get(userId);
        if (this.table.isTournamentSubTable() && (tournamentPlayer = this.table.getTournament().getPlayer(playerId)) != null) {
            tournamentPlayer.setStateInfo("");
        }
        if (this.table.isTournament() ? (player = this.tournament.getPlayer(playerId)) == null || player.hasQuit() : this.table.getMatch() != null && ((mPlayer2 = this.table.getMatch().getPlayer(playerId)) == null || mPlayer2.hasQuit())) {
            return true;
        }
        if (this.table.getState() != TableState.SIDEBOARDING && this.table.getState() != TableState.CONSTRUCTING) {
            return false;
        }
        Deck deck = Deck.load(deckList, false, false);
        if (this.table.getState() == TableState.SIDEBOARDING && this.table.getMatch() != null && (mPlayer = this.table.getMatch().getPlayer(playerId)) != null) {
            deck.setName(mPlayer.getDeck().getName());
        }
        if (!Main.isTestMode() && !this.table.getValidator().validate(deck)) {
            Optional<User> _user = this.managerFactory.userManager().getUser(userId);
            if (!_user.isPresent()) {
                return false;
            }
            StringBuilder sb = new StringBuilder("Invalid deck for the selected ").append(this.table.getValidator().getName()).append(" format. \n\n");
            List<DeckValidatorError> errorsList = this.table.getValidator().getErrorsListSorted();
            errorsList.stream().forEach(error -> sb.append(error.getGroup()).append(": ").append(error.getMessage()).append("\n"));
            sb.append("\n\nAdd enough cards and try again!");
            _user.get().showUserMessage("Submit deck", sb.toString());
            return false;
        }
        this.submitDeck(userId, playerId, deck);
        return true;
    }

    public void updateDeck(UUID userId, DeckCardLists deckList) throws MageException {
        UUID playerId = this.userPlayerMap.get(userId);
        if (this.table.getState() != TableState.SIDEBOARDING && this.table.getState() != TableState.CONSTRUCTING) {
            return;
        }
        Deck deck = Deck.load(deckList, false, false);
        this.updateDeck(userId, playerId, deck);
    }

    private void submitDeck(UUID userId, UUID playerId, Deck deck) {
        if (this.table.getState() == TableState.SIDEBOARDING) {
            this.match.submitDeck(playerId, deck);
            this.managerFactory.userManager().getUser(userId).ifPresent(user -> user.removeSideboarding(this.table.getId()));
        } else {
            this.managerFactory.tournamentManager().submitDeck(this.tournament.getId(), playerId, deck);
            this.managerFactory.userManager().getUser(userId).ifPresent(user -> user.removeConstructing(playerId));
        }
    }

    private void updateDeck(UUID userId, UUID playerId, Deck deck) {
        boolean ignoreMainBasicLands;
        boolean bl = ignoreMainBasicLands = this.table.isTournament() || this.table.isTournamentSubTable();
        if (this.table.isTournament()) {
            if (this.tournament != null) {
                this.managerFactory.tournamentManager().updateDeck(this.tournament.getId(), playerId, deck, ignoreMainBasicLands);
            } else {
                logger.fatal((Object)("Tournament == null  table: " + this.table.getId() + " userId: " + userId));
            }
        } else if (this.table.getState() == TableState.SIDEBOARDING) {
            this.match.updateDeck(playerId, deck, ignoreMainBasicLands);
        } else {
            logger.warn((Object)"wtf, why it submitting?!");
        }
    }

    public boolean watchTable(UUID userId) {
        if (this.table.isTournament()) {
            this.managerFactory.userManager().getUser(userId).ifPresent(user -> user.ccShowTournament(this.table.getId(), this.table.getTournament().getId()));
            return true;
        }
        if (this.table.isTournamentSubTable() && !this.table.getTournament().getOptions().isWatchingAllowed()) {
            return false;
        }
        if (this.table.getState() != TableState.DUELING) {
            return false;
        }
        if (this.userPlayerMap.get(userId) != null) {
            return false;
        }
        Optional<User> _user = this.managerFactory.userManager().getUser(userId);
        if (!_user.isPresent()) {
            return false;
        }
        return _user.get().ccWatchGame(this.table.getId(), this.table.getParentTableId(), this.match.getGame().getId());
    }

    private Optional<Player> createPlayer(String name, PlayerType playerType, int skill) {
        Optional<Player> playerOpt = this.options == null ? PlayerFactory.instance.createPlayer(playerType, name, RangeOfInfluence.ALL, skill) : PlayerFactory.instance.createPlayer(playerType, name, this.options.getRange(), skill);
        if (playerOpt.isPresent()) {
            Player player = playerOpt.get();
            logger.trace((Object)("Player " + player.getName() + " created id: " + player.getId()));
        }
        return playerOpt;
    }

    public void leaveTableAll() {
        for (UUID leavingUserId : this.userPlayerMap.keySet()) {
            this.leaveTable(leavingUserId);
        }
        this.closeTable();
    }

    public synchronized void leaveTable(UUID userId) {
        if (this.table == null) {
            logger.error((Object)("No table object - userId: " + userId));
            return;
        }
        if (this.table.isTournament() && this.tournament == null) {
            logger.error((Object)("No tournament object - userId: " + userId + "  table: " + this.table.getId()));
            return;
        }
        if (this.userId != null && this.userId.equals(userId) && (this.table.getState() == TableState.WAITING || this.table.getState() == TableState.READY_TO_START)) {
            this.managerFactory.tableManager().removeTable(this.table.getId());
        } else {
            UUID playerId = this.userPlayerMap.get(userId);
            if (playerId != null) {
                if (this.table.getState() == TableState.WAITING || this.table.getState() == TableState.READY_TO_START) {
                    this.table.leaveNotStartedTable(playerId);
                    if (this.table.isTournament()) {
                        this.tournament.removePlayer(playerId);
                    } else {
                        this.match.quitMatch(playerId);
                    }
                    Optional<User> user = this.managerFactory.userManager().getUser(userId);
                    if (user.isPresent()) {
                        this.managerFactory.chatManager().broadcast(this.chatId, user.get().getName(), "has left the table", ChatMessage.MessageColor.BLUE, true, null, ChatMessage.MessageType.STATUS, ChatMessage.SoundToPlay.PlayerLeft);
                        if (!this.table.isTournamentSubTable()) {
                            user.get().removeTable(playerId);
                        }
                    } else {
                        logger.debug((Object)("User not found - userId: " + userId + " tableId:" + this.table.getId()));
                    }
                    this.userPlayerMap.remove(userId);
                } else if (this.table.getState() != TableState.FINISHED) {
                    if (this.table.isTournament()) {
                        logger.debug((Object)("Quit tournament sub tables for userId: " + userId));
                        this.managerFactory.tableManager().userQuitTournamentSubTables(this.tournament.getId(), userId);
                        logger.debug((Object)("Quit tournament  Id: " + this.table.getTournament().getId() + '(' + this.table.getTournament().getTournamentState() + ')'));
                        this.managerFactory.tournamentManager().quit(this.tournament.getId(), userId);
                    } else {
                        MatchPlayer matchPlayer = this.match.getPlayer(playerId);
                        if (matchPlayer != null && !this.match.hasEnded() && !matchPlayer.hasQuit()) {
                            Game game = this.match.getGame();
                            if (game != null && !game.hasEnded()) {
                                Player player = this.match.getPlayer(playerId).getPlayer();
                                if (player != null && player.isInGame()) {
                                    this.managerFactory.gameManager().quitMatch(game.getId(), userId);
                                }
                                this.match.quitMatch(playerId);
                            } else {
                                if (this.table.getState() == TableState.SIDEBOARDING && !matchPlayer.isDoneSideboarding()) {
                                    matchPlayer.submitDeck(matchPlayer.getDeck());
                                }
                                this.match.quitMatch(playerId);
                            }
                        }
                    }
                }
            } else {
                logger.error((Object)("No playerId found for userId: " + userId));
            }
        }
    }

    public synchronized void startMatch(UUID userId) {
        if (this.isOwner(userId)) {
            this.startMatch();
        }
    }

    public synchronized void startMatch() {
        if (this.table.getState() == TableState.STARTING) {
            try {
                if (this.table.isTournamentSubTable()) {
                    logger.info((Object)("Tourney MATCH started id:" + this.match.getId() + " tournId: " + this.table.getTournament().getId()));
                } else {
                    this.managerFactory.userManager().getUser(this.userId).ifPresent(user -> {
                        logger.info((Object)("Single MATCH started [" + this.match.getName() + "] " + this.match.getId() + " (" + user.getName() + ')'));
                        logger.debug((Object)("- " + this.match.getOptions().getGameType() + " - " + this.match.getOptions().getDeckType()));
                    });
                }
                this.match.startMatch();
                this.startGame(null);
            }
            catch (GameException e) {
                logger.fatal((Object)("Error starting match: " + e), (Throwable)e);
                this.match.endGame();
            }
        }
    }

    private void startGame(UUID choosingPlayerId) throws GameException {
        block10: {
            try {
                this.match.startGame();
                this.table.initGame();
                GameOptions gameOptions = new GameOptions();
                gameOptions.rollbackTurnsAllowed = this.match.getOptions().isRollbackTurnsAllowed();
                gameOptions.bannedUsers = this.match.getOptions().getBannedUsers();
                gameOptions.planeChase = this.match.getOptions().isPlaneChase();
                gameOptions.perPlayerEmblemCards = this.match.getOptions().getPerPlayerEmblemCards();
                gameOptions.globalEmblemCards = this.match.getOptions().getGlobalEmblemCards();
                this.match.getGame().setGameOptions(gameOptions);
                this.managerFactory.gameManager().createGameSession(this.match.getGame(), this.userPlayerMap, this.table.getId(), choosingPlayerId, gameOptions);
                String creator = null;
                StringBuilder opponent = new StringBuilder();
                for (Map.Entry<UUID, UUID> entry : this.userPlayerMap.entrySet()) {
                    if (this.match.getPlayer(entry.getValue()) == null || this.match.getPlayer(entry.getValue()).hasQuit()) continue;
                    Optional<User> _user = this.managerFactory.userManager().getUser(entry.getKey());
                    if (_user.isPresent()) {
                        User user = _user.get();
                        user.ccGameStarted(this.table.getId(), this.table.getParentTableId(), this.match.getGame().getId(), entry.getValue());
                        if (creator == null) {
                            creator = user.getName();
                            continue;
                        }
                        if (opponent.length() > 0) {
                            opponent.append(" - ");
                        }
                        opponent.append(user.getName());
                        continue;
                    }
                    logger.error((Object)("Unable to find user: " + entry.getKey() + "  playerId: " + entry.getValue()));
                    MatchPlayer matchPlayer = this.match.getPlayer(entry.getValue());
                    if (matchPlayer == null || matchPlayer.hasQuit()) continue;
                    matchPlayer.setQuit(true);
                }
                for (MatchPlayer mPlayer : this.match.getPlayers()) {
                    if (mPlayer.getPlayer().isHuman()) continue;
                    if (opponent.length() > 0) {
                        opponent.append(" - ");
                    }
                    opponent.append(mPlayer.getName());
                }
                ServerMessagesUtil.instance.incGamesStarted();
                logger.info((Object)("GAME started " + (this.match.getGame() != null ? this.match.getGame().getId() : "no Game") + " [" + this.match.getName() + "] " + creator + " - " + opponent));
                logger.debug((Object)("- matchId: " + this.match.getId() + " [" + this.match.getName() + ']'));
                if (this.match.getGame() != null) {
                    logger.debug((Object)("- chatId:  " + this.managerFactory.gameManager().getChatId(this.match.getGame().getId())));
                }
            }
            catch (Exception ex) {
                Game game;
                logger.fatal((Object)("Error starting game table: " + this.table.getId()), (Throwable)ex);
                if (this.table != null) {
                    this.managerFactory.tableManager().removeTable(this.table.getId());
                }
                if (this.match == null || (game = this.match.getGame()) == null) break block10;
                this.managerFactory.gameManager().removeGame(game.getId());
            }
        }
    }

    public synchronized void startTournament(UUID userId) {
        try {
            if (userId.equals(this.userId) && this.table.getState() == TableState.STARTING) {
                this.tournament.setStartTime();
                this.managerFactory.tournamentManager().createTournamentSession(this.tournament, this.userPlayerMap, this.table.getId());
                for (Map.Entry<UUID, UUID> entry : this.userPlayerMap.entrySet()) {
                    this.managerFactory.userManager().getUser(entry.getKey()).ifPresent(user -> {
                        logger.info((Object)new StringBuilder("User ").append(user.getName()).append(" tournament started: ").append(this.tournament.getId()).append(" userId: ").append(user.getId()));
                        user.ccTournamentStarted(this.table.getId(), this.tournament.getId(), (UUID)entry.getValue());
                    });
                }
                ServerMessagesUtil.instance.incTournamentsStarted();
            }
        }
        catch (Exception ex) {
            logger.fatal((Object)"Error starting tournament", (Throwable)ex);
            this.managerFactory.tableManager().removeTable(this.table.getId());
            this.managerFactory.tournamentManager().quit(this.tournament.getId(), userId);
        }
    }

    public void startDraft(Draft draft) {
        this.table.initDraft(draft);
        this.managerFactory.draftManager().createDraftSession(draft, this.userPlayerMap, this.table.getId());
        for (Map.Entry<UUID, UUID> entry : this.userPlayerMap.entrySet()) {
            Optional<User> user = this.managerFactory.userManager().getUser(entry.getKey());
            if (user.isPresent()) {
                logger.info((Object)new StringBuilder("User ").append(user.get().getName()).append(" draft started: ").append(draft.getId()).append(" userId: ").append(user.get().getId()));
                user.get().ccDraftStarted(this.table.getId(), draft.getId(), entry.getValue());
                continue;
            }
            logger.fatal((Object)new StringBuilder("Start draft user not found userId: ").append(entry.getKey()));
        }
    }

    private void sideboard(UUID playerId, Deck deck) throws MageException {
        for (Map.Entry<UUID, UUID> entry : this.userPlayerMap.entrySet()) {
            if (!entry.getValue().equals(playerId)) continue;
            Optional<User> user = this.managerFactory.userManager().getUser(entry.getKey());
            int remaining = (int)this.futureTimeout.getDelay(TimeUnit.SECONDS);
            user.ifPresent(user1 -> user1.ccSideboard(deck, this.table.getId(), this.table.getParentTableId(), remaining, this.options.isLimited()));
            break;
        }
    }

    public int getRemainingTime() {
        return (int)this.futureTimeout.getDelay(TimeUnit.SECONDS);
    }

    public void construct() {
        this.table.construct();
    }

    public void initTournament() {
        this.table.initTournament();
    }

    public void endTournament(Tournament tournament) {
        this.table.endTournament();
        ServerMessagesUtil.instance.incTournamentsEnded();
    }

    public MatchOptions getOptions() {
        return this.options;
    }

    public boolean endGameAndStartNextGame() {
        Game game = this.match.getGame();
        if (game == null) {
            return true;
        }
        UUID choosingPlayerId = this.match.getChooser();
        this.match.endGame();
        if (this.managerFactory.configSettings().isSaveGameActivated().booleanValue() && !game.isSimulation() && this.managerFactory.gameManager().saveGame(game.getId())) {
            this.match.setReplayAvailable(true);
        }
        this.managerFactory.gameManager().removeGame(game.getId());
        ServerMessagesUtil.instance.incGamesEnded();
        try {
            if (!this.match.hasEnded()) {
                if (this.match.getGame() != null && this.match.getGame().getGameType().isSideboardingAllowed()) {
                    this.sideboard();
                }
                if (!this.match.hasEnded()) {
                    this.startGame(choosingPlayerId);
                } else {
                    this.closeTable();
                }
            } else {
                this.closeTable();
            }
        }
        catch (GameException ex) {
            logger.fatal(null, (Throwable)ex);
        }
        return this.match.hasEnded();
    }

    private void sideboard() {
        TournamentPlayer tournamentPlayer;
        this.table.sideboard();
        this.setupTimeout(180);
        if (this.table.isTournamentSubTable()) {
            for (MatchPlayer matchPlayer : this.match.getPlayers()) {
                if (matchPlayer.hasQuit() || (tournamentPlayer = this.table.getTournament().getPlayer(matchPlayer.getPlayer().getId())) == null) continue;
                tournamentPlayer.setStateInfo("sideboarding");
            }
        }
        this.match.sideboard();
        this.cancelTimeout();
        if (this.table.isTournamentSubTable()) {
            for (MatchPlayer matchPlayer : this.match.getPlayers()) {
                tournamentPlayer = this.table.getTournament().getPlayer(matchPlayer.getPlayer().getId());
                if (tournamentPlayer == null || !tournamentPlayer.getStateInfo().equals("sideboarding")) continue;
                tournamentPlayer.setStateInfo("");
            }
        }
    }

    private void closeTable() {
        this.matchEnd();
        this.table.closeTable();
    }

    private void matchEnd() {
        if (this.match != null) {
            for (Map.Entry<UUID, UUID> entry : this.userPlayerMap.entrySet()) {
                MatchPlayer matchPlayer = this.match.getPlayer(entry.getValue());
                if (matchPlayer == null || matchPlayer.hasQuit()) continue;
                this.managerFactory.userManager().getUser(entry.getKey()).ifPresent(user -> {
                    if (this.table.getState() == TableState.SIDEBOARDING) {
                        StringBuilder sb = new StringBuilder();
                        if (this.table.isTournamentSubTable()) {
                            sb.append("Your tournament match of round ");
                            sb.append(this.table.getTournament().getRounds().size());
                            sb.append(" is over. ");
                        } else {
                            sb.append("Match [").append(this.match.getName()).append("] is over. ");
                        }
                        if (this.match.getPlayers().size() > 2) {
                            sb.append("All your opponents have lost or quit the match.");
                        } else {
                            sb.append("Your opponent has quit the match.");
                        }
                        user.showUserMessage("Match info", sb.toString());
                    }
                    if (!this.table.isTournamentSubTable()) {
                        user.removeTable((UUID)entry.getValue());
                    }
                });
            }
            this.match.cleanUpOnMatchEnd(this.managerFactory.configSettings().isSaveGameActivated(), this.table.isTournament());
        }
    }

    private synchronized void setupTimeout(int seconds) {
        this.cancelTimeout();
        if (seconds > 0) {
            this.futureTimeout = this.timeoutExecutor.schedule(this::autoSideboard, (long)seconds, TimeUnit.SECONDS);
        }
    }

    private synchronized void cancelTimeout() {
        if (this.futureTimeout != null) {
            this.futureTimeout.cancel(false);
        }
    }

    private void autoSideboard() {
        for (MatchPlayer player : this.match.getPlayers()) {
            if (player.isDoneSideboarding()) continue;
            this.match.submitDeck(player.getPlayer().getId(), player.autoCompleteDeck(this.table.getValidator()));
        }
    }

    public void endDraft(Draft draft) {
        if (!draft.isAbort()) {
            for (DraftPlayer player : draft.getPlayers()) {
                player.prepareDeck();
                this.tournament.getPlayer(player.getPlayer().getId()).setDeck(player.getDeck());
            }
        }
        this.tournament.clearDraft();
        this.tournament.nextStep();
    }

    public void swapSeats(int seatNum1, int seatNum2) {
        if (this.table.getState() == TableState.READY_TO_START && seatNum1 >= 0 && seatNum2 >= 0 && seatNum1 < this.table.getSeats().length && seatNum2 < this.table.getSeats().length) {
            Player swapPlayer = this.table.getSeats()[seatNum1].getPlayer();
            PlayerType swapType = this.table.getSeats()[seatNum1].getPlayerType();
            this.table.getSeats()[seatNum1].setPlayer(this.table.getSeats()[seatNum2].getPlayer());
            this.table.getSeats()[seatNum1].setPlayerType(this.table.getSeats()[seatNum2].getPlayerType());
            this.table.getSeats()[seatNum2].setPlayer(swapPlayer);
            this.table.getSeats()[seatNum2].setPlayerType(swapType);
        }
    }

    public boolean isOwner(UUID userId) {
        if (userId == null) {
            return false;
        }
        return userId.equals(this.userId);
    }

    public Table getTable() {
        return this.table;
    }

    public UUID getChatId() {
        return this.chatId;
    }

    public Match getMatch() {
        return this.match;
    }

    public boolean isTournamentStillValid() {
        if (this.table.getTournament() != null) {
            if (this.table.getState() != TableState.WAITING && this.table.getState() != TableState.READY_TO_START && this.table.getState() != TableState.STARTING) {
                return this.managerFactory.tournamentManager().getTournamentController(this.table.getTournament().getId()).map(tc -> tc.isTournamentStillValid(this.table.getState())).orElse(false);
            }
            return this.managerFactory.userManager().getUser(this.userId).isPresent();
        }
        return false;
    }

    public UUID getUserId(UUID playerId) {
        for (Map.Entry<UUID, UUID> entry : this.userPlayerMap.entrySet()) {
            if (!entry.getValue().equals(playerId)) continue;
            return entry.getKey();
        }
        return null;
    }

    public boolean isUserStillActive(UUID userId) {
        UUID playerId = this.userPlayerMap.get(userId);
        if (playerId != null) {
            if (this.tournament != null) {
                TournamentPlayer tournamentPlayer = this.tournament.getPlayer(playerId);
                if (tournamentPlayer != null) {
                    return tournamentPlayer.isInTournament();
                }
            } else if (this.match != null) {
                MatchPlayer matchPlayer = this.match.getPlayer(playerId);
                return matchPlayer != null && !matchPlayer.hasQuit();
            }
        }
        return false;
    }

    public boolean isMatchTableStillValid() {
        if (this.table.isTournament()) {
            return true;
        }
        if (Arrays.asList(TableState.WAITING, TableState.READY_TO_START, TableState.STARTING).contains((Object)this.table.getState())) {
            return true;
        }
        if (this.match != null && !this.match.isDoneSideboarding()) {
            return true;
        }
        if (this.match.getGame() == null) {
            logger.error((Object)"- Match without started games:");
            logger.error((Object)("-- matchId:" + this.match.getId()));
            return false;
        }
        int validHumanPlayers = 0;
        int validAIPlayers = 0;
        for (Map.Entry<UUID, UUID> userPlayerEntry : this.userPlayerMap.entrySet()) {
            MatchPlayer matchPlayer = this.match.getPlayer(userPlayerEntry.getValue());
            if (matchPlayer == null) {
                logger.error((Object)"- Match player not found in started game:");
                logger.error((Object)("-- matchId:" + this.match.getId()));
                logger.error((Object)("-- userId:" + userPlayerEntry.getKey()));
                logger.error((Object)("-- playerId:" + userPlayerEntry.getValue()));
                continue;
            }
            if (!matchPlayer.getPlayer().isHuman() || !matchPlayer.getPlayer().isInGame()) continue;
            Optional<User> user = this.managerFactory.userManager().getUser(userPlayerEntry.getKey());
            if (!user.isPresent() || !user.get().isActive()) {
                logger.error((Object)("- Active user of match is missing: " + matchPlayer.getName()));
                logger.error((Object)("-- matchId:" + this.match.getId()));
                logger.error((Object)("-- userId:" + userPlayerEntry.getKey()));
                logger.error((Object)("-- playerId:" + userPlayerEntry.getValue()));
                return false;
            }
            ++validHumanPlayers;
        }
        for (MatchPlayer matchPlayer : this.match.getPlayers()) {
            if (matchPlayer.getPlayer().isHuman() || !matchPlayer.getPlayer().isInGame()) continue;
            ++validAIPlayers;
        }
        return validAIPlayers + validHumanPlayers >= 2;
    }

    void cleanUp() {
        if (!this.table.isTournamentSubTable()) {
            for (Map.Entry<UUID, UUID> entry : this.userPlayerMap.entrySet()) {
                this.managerFactory.userManager().getUser(entry.getKey()).ifPresent(user -> user.removeTable((UUID)entry.getValue()));
            }
        }
        this.managerFactory.chatManager().destroyChatSession(this.chatId);
    }

    public synchronized TableState getTableState() {
        return this.table.getState();
    }

    public synchronized boolean changeTableStateToStarting() {
        if (this.table.getState() != TableState.READY_TO_START) {
            return false;
        }
        if (!this.table.allSeatsAreOccupied()) {
            logger.debug((Object)("Not alle Seats are occupied: stop start tableId:" + this.table.getId()));
            return false;
        }
        this.table.setState(TableState.STARTING);
        return true;
    }
}

