/*
 * Decompiled with CFR 0.152.
 */
package mage.server.tournament;

import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import mage.MageException;
import mage.cards.decks.Deck;
import mage.cards.decks.DeckValidator;
import mage.cards.decks.DeckValidatorFactory;
import mage.constants.TableState;
import mage.constants.TournamentPlayerState;
import mage.game.GameException;
import mage.game.Table;
import mage.game.draft.Draft;
import mage.game.match.Match;
import mage.game.match.MatchOptions;
import mage.game.result.ResultProtos;
import mage.game.tournament.MultiplayerRound;
import mage.game.tournament.Tournament;
import mage.game.tournament.TournamentPairing;
import mage.game.tournament.TournamentPlayer;
import mage.players.PlayerType;
import mage.server.User;
import mage.server.draft.DraftController;
import mage.server.managers.ManagerFactory;
import mage.server.managers.TableManager;
import mage.server.tournament.TournamentSession;
import mage.view.ChatMessage;
import mage.view.TournamentView;
import org.apache.log4j.Logger;

public class TournamentController {
    private static final Logger logger = Logger.getLogger(TournamentController.class);
    private final ManagerFactory managerFactory;
    private final UUID chatId;
    private final UUID tableId;
    private boolean started = false;
    private final Tournament tournament;
    private ConcurrentMap<UUID, UUID> userPlayerMap = new ConcurrentHashMap<UUID, UUID>();
    private final ConcurrentMap<UUID, TournamentSession> tournamentSessions = new ConcurrentHashMap<UUID, TournamentSession>();

    public TournamentController(ManagerFactory managerFactory, Tournament tournament, ConcurrentMap<UUID, UUID> userPlayerMap, UUID tableId) {
        this.managerFactory = managerFactory;
        this.userPlayerMap = userPlayerMap;
        this.chatId = managerFactory.chatManager().createTourneyChatSession(tournament);
        this.tournament = tournament;
        this.tableId = tableId;
        this.init();
    }

    private void init() {
        this.tournament.addTableEventListener(event -> {
            switch (event.getEventType()) {
                case CHECK_STATE_PLAYERS: {
                    this.checkPlayersState();
                    break;
                }
                case INFO: {
                    this.managerFactory.chatManager().broadcast(this.chatId, "", event.getMessage(), ChatMessage.MessageColor.BLACK, true, null, ChatMessage.MessageType.STATUS, null);
                    logger.debug((Object)(this.tournament.getId() + " " + event.getMessage()));
                    break;
                }
                case START_DRAFT: {
                    this.startDraft(event.getDraft());
                    break;
                }
                case CONSTRUCT: {
                    if (!this.isAbort()) {
                        this.construct();
                        break;
                    }
                    this.endTournament();
                    break;
                }
                case START_MATCH: {
                    if (this.isAbort()) break;
                    this.initTournament();
                    this.startMatch(event.getPair(), event.getMatchOptions());
                    break;
                }
                case START_MULTIPLAYER_MATCH: {
                    if (this.isAbort()) break;
                    this.initTournament();
                    MatchOptions matchOptions = event.getMatchOptions();
                    if (matchOptions != null && event.getMultiplayerRound() != null) {
                        for (TournamentPlayer player : event.getMultiplayerRound().getAllPlayers()) {
                            matchOptions.getPlayerTypes().add(player.getPlayerType());
                        }
                    }
                    this.startMultiplayerMatch(event.getMultiplayerRound(), event.getMatchOptions());
                    break;
                }
                case END: {
                    this.endTournament();
                }
            }
        });
        this.tournament.addPlayerQueryEventListener(event -> {
            try {
                switch (event.getQueryType()) {
                    case TOURNAMENT_CONSTRUCT: {
                        this.construct(event.getPlayerId(), event.getMax());
                    }
                }
            }
            catch (MageException ex) {
                logger.fatal((Object)"Player event listener error", (Throwable)ex);
            }
        });
        for (TournamentPlayer player : this.tournament.getPlayers()) {
            if (player.getPlayer().isHuman()) continue;
            player.setJoined();
            logger.debug((Object)("player " + player.getPlayer().getId() + " has joined tournament " + this.tournament.getId()));
            this.managerFactory.chatManager().broadcast(this.chatId, "", player.getPlayer().getLogName() + " has joined the tournament", ChatMessage.MessageColor.BLACK, true, null, ChatMessage.MessageType.STATUS, null);
        }
        this.checkStart();
    }

    public void join(UUID userId) {
        UUID playerId = (UUID)this.userPlayerMap.get(userId);
        if (playerId == null) {
            if (logger.isDebugEnabled()) {
                this.managerFactory.userManager().getUser(userId).ifPresent(user -> logger.debug((Object)(user.getName() + " shows tournament panel  tournamentId: " + this.tournament.getId())));
            }
            return;
        }
        if (this.tournamentSessions.containsKey(playerId)) {
            logger.debug((Object)("player reopened tournament panel userId: " + userId + " tournamentId: " + this.tournament.getId()));
            return;
        }
        User user2 = this.managerFactory.userManager().getUser(userId).orElse(null);
        if (user2 == null) {
            logger.error((Object)("User not found  userId: " + userId + "   tournamentId: " + this.tournament.getId()));
            return;
        }
        TournamentSession tournamentSession = new TournamentSession(this.managerFactory, this.tournament, userId, this.tableId, playerId);
        this.tournamentSessions.put(playerId, tournamentSession);
        user2.addTournament(playerId, this.tournament.getId());
        TournamentPlayer player = this.tournament.getPlayer(playerId);
        player.setJoined();
        logger.debug((Object)("player " + player.getPlayer().getName() + " - client has joined tournament " + this.tournament.getId()));
        this.managerFactory.chatManager().broadcast(this.chatId, "", player.getPlayer().getLogName() + " has joined the tournament", ChatMessage.MessageColor.BLACK, true, null, ChatMessage.MessageType.STATUS, null);
        this.checkStart();
    }

    public void rejoin(UUID playerId) {
        TournamentSession tournamentSession = (TournamentSession)this.tournamentSessions.get(playerId);
        if (tournamentSession == null) {
            logger.fatal((Object)("Tournament session not found - playerId:" + playerId + "  tournamentId " + this.tournament.getId()));
            return;
        }
        if (!tournamentSession.init()) {
            logger.fatal((Object)("Unable to initialize client userId: " + tournamentSession.userId + "  tournamentId " + this.tournament.getId()));
            return;
        }
        tournamentSession.update();
    }

    private synchronized void checkStart() {
        if (!this.started && this.allJoined()) {
            this.managerFactory.threadExecutor().getTourneyExecutor().execute(this::startTournament);
        }
    }

    private boolean allJoined() {
        if (!this.tournament.allJoined()) {
            return false;
        }
        for (TournamentPlayer player : this.tournament.getPlayers()) {
            if (!player.getPlayer().isHuman() || this.tournamentSessions.get(player.getPlayer().getId()) != null) continue;
            return false;
        }
        return true;
    }

    private void startTournament() {
        Thread.currentThread().setName("TOURNEY " + this.tableId);
        for (TournamentSession tournamentSession : this.tournamentSessions.values()) {
            if (tournamentSession.init()) continue;
            logger.fatal((Object)("Unable to initialize client userId: " + tournamentSession.userId + "  tournamentId " + this.tournament.getId()));
            return;
        }
        this.started = true;
        logger.debug((Object)("Tournament starts (all players joined): " + this.tournament.getId() + " - " + this.tournament.getTournamentType().toString()));
        this.tournament.nextStep();
    }

    private void endTournament() {
        boolean setFinishPlayersStatus = true;
        if (this.tournament.getRounds().isEmpty() && this.tournament.getOptions().getMatchOptions().isSingleGameTourney()) {
            setFinishPlayersStatus = false;
        }
        if (setFinishPlayersStatus) {
            for (TournamentPlayer player : this.tournament.getPlayers()) {
                player.setStateAtTournamentEnd();
            }
        }
        for (TournamentSession tournamentSession : this.tournamentSessions.values()) {
            tournamentSession.tournamentOver();
        }
        this.tournamentSessions.clear();
        this.managerFactory.tableManager().endTournament(this.tableId, this.tournament);
        this.tournament.cleanUpOnTournamentEnd();
    }

    private void startMatch(TournamentPairing pair, MatchOptions matchOptions) {
        try {
            TableManager tableManager = this.managerFactory.tableManager();
            Table table = tableManager.createTable(this.managerFactory.gamesRoomManager().getMainRoomId(), matchOptions);
            table.setTournamentSubTable(this.tableId);
            table.setTournament(this.tournament);
            table.setState(TableState.WAITING);
            TournamentPlayer player1 = pair.getPlayer1();
            TournamentPlayer player2 = pair.getPlayer2();
            UUID user1Uuid = null;
            UUID user2Uuid = null;
            if (player1.getPlayerType() == PlayerType.HUMAN) {
                Optional<UUID> user1Id = this.getPlayerUserId(player1.getPlayer().getId());
                if (!user1Id.isPresent()) {
                    logger.fatal((Object)"Player 1 not found");
                } else {
                    user1Uuid = user1Id.get();
                }
            }
            if (player2.getPlayerType() == PlayerType.HUMAN) {
                Optional<UUID> user2Id = this.getPlayerUserId(player2.getPlayer().getId());
                if (!user2Id.isPresent()) {
                    logger.fatal((Object)"Player 2 not found");
                } else {
                    user2Uuid = user2Id.get();
                }
            }
            tableManager.addPlayer(user1Uuid, table.getId(), player1);
            tableManager.addPlayer(user2Uuid, table.getId(), player2);
            table.setState(TableState.STARTING);
            tableManager.startTournamentSubMatch(null, table.getId());
            tableManager.getMatch(table.getId()).ifPresent(match -> {
                pair.setMatchAndTable((Match)match, table.getId());
                player1.setState(TournamentPlayerState.DUELING);
                player2.setState(TournamentPlayerState.DUELING);
            });
        }
        catch (GameException ex) {
            logger.fatal((Object)"TournamentController startMatch error", (Throwable)ex);
        }
    }

    private void startMultiplayerMatch(MultiplayerRound round, MatchOptions matchOptions) {
        try {
            TableManager tableManager = this.managerFactory.tableManager();
            Table table = tableManager.createTable(this.managerFactory.gamesRoomManager().getMainRoomId(), matchOptions);
            table.setTournamentSubTable(this.tableId);
            table.setTournament(this.tournament);
            table.setState(TableState.WAITING);
            if (round.getAllPlayers().stream().filter(t -> t.getPlayerType().equals((Object)PlayerType.HUMAN)).allMatch(t -> this.getPlayerUserId(t.getPlayer().getId()).isPresent())) {
                for (TournamentPlayer player : round.getAllPlayers()) {
                    UUID userId = this.getPlayerUserId(player.getPlayer().getId()).orElse(null);
                    tableManager.addPlayer(userId, table.getId(), player);
                }
                table.setState(TableState.STARTING);
                tableManager.startTournamentSubMatch(null, table.getId());
                tableManager.getMatch(table.getId()).ifPresent(match -> {
                    round.setMatchAndTable((Match)match, table.getId());
                    for (TournamentPlayer player : round.getAllPlayers()) {
                        player.setState(TournamentPlayerState.DUELING);
                    }
                });
            } else {
                logger.error((Object)"tourney - startMultiplayerMatch can't start due disconnected players");
            }
        }
        catch (GameException ex) {
            logger.fatal((Object)"tourney - startMultiplayerMatch error", (Throwable)ex);
        }
    }

    private void startDraft(Draft draft) {
        this.managerFactory.tableManager().startDraft(this.tableId, draft);
    }

    private void construct() {
        this.managerFactory.tableManager().construct(this.tableId);
    }

    private void initTournament() {
        if (this.managerFactory.tableManager().getTable(this.tableId).getState() != TableState.DUELING) {
            this.managerFactory.tableManager().initTournament(this.tableId);
        }
    }

    private void construct(UUID playerId, int timeout) throws MageException {
        if (this.tournamentSessions.containsKey(playerId)) {
            TournamentSession tournamentSession = (TournamentSession)this.tournamentSessions.get(playerId);
            tournamentSession.construct(timeout);
            this.getPlayerUserId(playerId).ifPresent(userId -> this.managerFactory.userManager().getUser((UUID)userId).ifPresent(user -> {
                user.addConstructing(playerId, tournamentSession);
                TournamentPlayer player = this.tournament.getPlayer(playerId);
                player.setState(TournamentPlayerState.CONSTRUCTING);
            }));
        }
    }

    public void submitDeck(UUID playerId, Deck deck) {
        TournamentPlayer player;
        if (this.tournamentSessions.containsKey(playerId) && (player = this.tournament.getPlayer(playerId)) != null && !player.hasQuit()) {
            ((TournamentSession)this.tournamentSessions.get(playerId)).submitDeck(deck);
            this.managerFactory.chatManager().broadcast(this.chatId, "", player.getPlayer().getLogName() + " has submitted their tournament deck", ChatMessage.MessageColor.BLACK, true, null, ChatMessage.MessageType.STATUS, ChatMessage.SoundToPlay.PlayerSubmittedDeck);
        }
    }

    public void updateDeck(UUID playerId, Deck deck, boolean ignoreMainBasicLands) {
        TournamentSession session = this.tournamentSessions.getOrDefault(playerId, null);
        if (session == null) {
            return;
        }
        session.updateDeck(deck, ignoreMainBasicLands);
    }

    public void timeout(UUID userId) {
        if (this.userPlayerMap.containsKey(userId)) {
            TournamentPlayer tournamentPlayer = this.tournament.getPlayer((UUID)this.userPlayerMap.get(userId));
            if (tournamentPlayer.getDeck() != null) {
                DeckValidator deckValidator = DeckValidatorFactory.instance.createDeckValidator(this.tournament.getOptions().getMatchOptions().getDeckType());
                int deckMinSize = deckValidator != null ? deckValidator.getDeckMinSize() : 40;
                this.tournament.autoSubmit((UUID)this.userPlayerMap.get(userId), tournamentPlayer.generateDeck(deckMinSize));
            } else {
                StringBuilder sb = new StringBuilder();
                this.managerFactory.userManager().getUser(userId).ifPresent(user -> sb.append(user.getName()));
                sb.append(" - no deck found for auto submit");
                logger.fatal((Object)sb);
                tournamentPlayer.setEliminated();
                tournamentPlayer.setStateInfo("No deck for auto submit");
            }
        }
    }

    public UUID getChatId() {
        return this.chatId;
    }

    public void quit(UUID userId) {
        UUID playerId = (UUID)this.userPlayerMap.get(userId);
        if (playerId == null) {
            logger.debug((Object)("Player not found userId:" + userId + " tournId: " + this.tournament.getId()));
            return;
        }
        TournamentPlayer tournamentPlayer = this.tournament.getPlayer(playerId);
        if (tournamentPlayer == null) {
            logger.debug((Object)("TournamentPlayer not found userId: " + userId + " tournId: " + this.tournament.getId()));
            return;
        }
        if (!this.started) {
            this.tournament.leave(playerId);
            return;
        }
        TournamentSession tournamentSession = (TournamentSession)this.tournamentSessions.get(playerId);
        if (tournamentSession == null) {
            logger.debug((Object)("TournamentSession not found userId: " + userId + " tournId: " + this.tournament.getId()));
            return;
        }
        tournamentSession.setKilled();
        if (tournamentPlayer.isInTournament()) {
            ResultProtos.TourneyQuitStatus status;
            String info;
            if (this.tournament.isDoneConstructing()) {
                info = "during round " + this.tournament.getRounds().size();
                this.managerFactory.tableManager().userQuitTournamentSubTables(this.tournament.getId(), userId);
                status = ResultProtos.TourneyQuitStatus.DURING_ROUND;
            } else if (tournamentPlayer.getState() == TournamentPlayerState.DRAFTING) {
                info = "during Draft phase";
                if (!this.checkToReplaceDraftPlayerByAi(userId, tournamentPlayer)) {
                    this.abortDraftTournament();
                } else {
                    this.managerFactory.draftManager().getController(this.tableId).ifPresent(draftController -> draftController.getDraftSession(playerId).ifPresent(draftSession -> this.managerFactory.draftManager().kill(draftSession.getDraft().getId(), userId)));
                }
                status = ResultProtos.TourneyQuitStatus.DURING_DRAFTING;
            } else if (tournamentPlayer.getState() == TournamentPlayerState.CONSTRUCTING) {
                info = "during Construction phase";
                status = ResultProtos.TourneyQuitStatus.DURING_CONSTRUCTION;
            } else {
                info = "";
                status = ResultProtos.TourneyQuitStatus.NO_TOURNEY_QUIT;
            }
            tournamentPlayer.setQuit(info, status);
            this.tournament.quit(playerId);
            tournamentSession.quit();
            this.managerFactory.chatManager().broadcast(this.chatId, "", tournamentPlayer.getPlayer().getLogName() + " has quit the tournament", ChatMessage.MessageColor.BLACK, true, null, ChatMessage.MessageType.STATUS, ChatMessage.SoundToPlay.PlayerQuitTournament);
        }
    }

    private boolean checkToReplaceDraftPlayerByAi(UUID userId, TournamentPlayer leavingPlayer) {
        int humans = 0;
        for (TournamentPlayer tPlayer : this.tournament.getPlayers()) {
            if (!tPlayer.getPlayer().isHuman()) continue;
            ++humans;
        }
        if (humans > 1) {
            Optional<User> user = this.managerFactory.userManager().getUser(userId);
            this.managerFactory.tableManager().getController(this.tableId).ifPresent(tableController -> {
                String replacePlayerName = "Draftbot";
                if (user.isPresent()) {
                    replacePlayerName = "Draftbot (" + ((User)user.get()).getName() + ')';
                }
                tableController.replaceDraftPlayer(leavingPlayer.getPlayer(), replacePlayerName, PlayerType.COMPUTER_DRAFT_BOT, 5);
                if (user.isPresent()) {
                    ((User)user.get()).removeDraft(leavingPlayer.getPlayer().getId());
                    ((User)user.get()).removeTable(leavingPlayer.getPlayer().getId());
                    ((User)user.get()).removeTournament(leavingPlayer.getPlayer().getId());
                }
                this.managerFactory.chatManager().broadcast(this.chatId, "", leavingPlayer.getPlayer().getLogName() + " was replaced by draftbot", ChatMessage.MessageColor.BLACK, true, null, ChatMessage.MessageType.STATUS, null);
            });
            return true;
        }
        return false;
    }

    private Optional<UUID> getPlayerUserId(UUID playerId) {
        return this.userPlayerMap.entrySet().stream().filter(entry -> ((UUID)entry.getValue()).equals(playerId)).map(Map.Entry::getKey).findFirst();
    }

    public TournamentView getTournamentView() {
        return new TournamentView(this.tournament);
    }

    private void abortDraftTournament() {
        this.tournament.setAbort(true);
        this.managerFactory.draftManager().getController(this.tableId).ifPresent(DraftController::abortDraft);
    }

    public boolean isAbort() {
        return this.tournament.isAbort();
    }

    private void checkPlayersState() {
        for (TournamentPlayer tournamentPlayer : this.tournament.getPlayers()) {
            if (tournamentPlayer.isEliminated() || !tournamentPlayer.getPlayer().isHuman()) continue;
            if (this.tournamentSessions.containsKey(tournamentPlayer.getPlayer().getId())) {
                if (!((TournamentSession)this.tournamentSessions.get(tournamentPlayer.getPlayer().getId())).isKilled()) continue;
                tournamentPlayer.setEliminated();
                tournamentPlayer.setStateInfo("disconnected");
                continue;
            }
            tournamentPlayer.setEliminated();
            tournamentPlayer.setStateInfo("no tournament session");
        }
    }

    public void cleanUpOnRemoveTournament() {
        this.managerFactory.chatManager().destroyChatSession(this.chatId);
    }

    public boolean isTournamentStillValid(TableState tableState) {
        int activePlayers = 0;
        for (Map.Entry entry : this.userPlayerMap.entrySet()) {
            TournamentPlayer tournamentPlayer = this.tournament.getPlayer((UUID)entry.getValue());
            if (tournamentPlayer != null) {
                if (tournamentPlayer.hasQuit() || !tournamentPlayer.getPlayer().isHuman()) continue;
                Optional<User> user = this.managerFactory.userManager().getUser((UUID)entry.getKey());
                if (!user.isPresent()) {
                    logger.debug((Object)("Tournament user is missing but player active -> start quit - tournamentId: " + this.tournament.getId() + " state: " + tableState.toString()));
                    this.quit((UUID)entry.getKey());
                    continue;
                }
                ++activePlayers;
                continue;
            }
            logger.debug((Object)("Tournament player is missing - tournamentId: " + this.tournament.getId() + " state: " + tableState.toString()));
        }
        for (TournamentPlayer tournamentPlayer : this.tournament.getPlayers()) {
            if (tournamentPlayer.getPlayer().isHuman() || tournamentPlayer.hasQuit()) continue;
            ++activePlayers;
        }
        if (activePlayers < 2 && tableState != TableState.WAITING) {
            logger.debug((Object)("Tournament has less than 2 active players - tournamentId: " + this.tournament.getId() + " state: " + tableState.toString()));
            return false;
        }
        return true;
    }

    public UUID getTableId() {
        return this.tableId;
    }
}

